﻿ <#
.SYNOPSIS

        Change the auto-delegate setting for an arbitrary user

.DESCRIPTION

        The FileHoldApi.psm1 helper module should be in the same folder as this script.
        If you leave out username and password, Windows Integrated Authentication will
        use the credentials of the current user.

.NOTES

        Copyright(C)2022 FileHold Systems Inc.

        *** DISCLAIMER ***

        FileHold makes no claims to the correctness, fitness for purpose, or
        anything else related to this script. It is provided as an example only.
        It is intended to be used or modified by a person skilled with Windows,
        PowerShell, .NET programming, and the FileHold API. Never use it on a
        production system without thouroughly testing it first and never use it
        in production if you do not fit the skilled person description above.

        *** DISCLAIMER ***

.EXAMPLE set new delegate

    Set-FileHoldAutoDelegateUser -Server 'http://myserver' -UserId 'sysadm' -Password '12345' -SearchType ExactName -FromUser 'user one' -ToUser sysadm
     
.EXAMPLE remove delegate

    Set-FileHoldAutoDelegateUser -Server 'http://myserver' -UserId 'sysadm' -Password '12345' -SearchType ExactName -FromUser 'user one'

#>

$GetFileHoldAutoDelegateUserBuildNumber = '20220406.1'

function Get-FileHoldAutoDelegateUser
{
    [CmdletBinding(SupportsShouldProcess)] # SupportsShouldProcess needed for -WhatIf and -Confirm
    Param (
        [Parameter(ValueFromPipeline,Mandatory,HelpMessage='List of search values'
        )][Object]$ValueList,

        [Parameter(HelpMessage='Name or URI for FileHold server. HTTP assumed if protocol not specified.'
        )][uri]$Server = 'localhost',
    
        [Parameter(HelpMessage='FileHold user id with sufficient roll or null for IWA.'
        )][string]$UserId = $null,
    
        [Parameter(HelpMessage='Password for given user or null for IWA.'
        )][string]$Password = $null,
    
        [Parameter(HelpMessage='Active directory domain name for given user.'
        )][string]$Domain = $null,

        [Parameter(HelpMessage='Set to not log off the at the end of the script.'
        )][switch]$PreserveSession = $false,

        [Parameter(HelpMessage='Set to use the current session cache instead of a new log in.'
        )][switch]$ReuseSession= $false,

        [Parameter(HelpMessage='Set this switch to prompt for credentials.'
        )][switch]$PromptForAuthentication = $false,

        [Parameter(HelpMessage='Set this to the method to search the user list.'
        )][ValidateSet( 'ExactName', 'ExactEmail', 'StartsWithName' )]
        [string]$SearchType = 'ExactName'
    )
  
    Begin
    {
        Write-Information -MessageData "Get auto-delegation for user $(Get-Date) ($($MyInvocation.MyCommand.Name) $GetFileHoldAutoDelegateUserBuildNumber)"

        if ( $WhatIfPreference ) 
        {
            Write-Verbose -Message 'Simulate mode is enabled. The system will not be updated. Invoke without ""-WhatIf"" to make changes.'
        }
    
        if ( !$ReuseSession )    
        {
            try {
                Start-FileHoldSession    -HostAddress $Server `
                                            -UserId $UserId `
                                            -Password $Password `
                                            -Domain $Domain `
                                            -ClientName "($($MyInvocation.MyCommand.Name) $BuildNumber)" `
                                            -ClientVersion '16.2' `
                                            -InformationAction $InformationPreference `
                                            -PromptForAuthentication:$PromptForAuthentication
            } catch {
                Write-Error $_
                exit
            }
        }
    }      

    Process
    {
        foreach ( $record in $ValueList )
        {
            try {
                if ( $pscmdlet.ShouldProcess( $record, "Get user guid" ))
                {
                    $userObjects = Get-FileHoldUserObjects -ReuseSession -PreserveSession -ValueList $record
                    foreach  ( $userObject in $userObjects )
                    {
                        if (( $userObject.Type -eq 0 -or $userObject.Type -eq 1 ) `
                                -and ( $userObject.HasFHAccountEnabled ) `
                                -and ( $userObject.RoleType -eq 0 ))
                        {
                            $userSettings = (UserPreferences).GetUserWorkflowNotificationsSettings( $userObject.ID )
                            if ( $userSettings.DelegateToUser -ne $null )
                            {
                                $delegatedUserObjects = (UserGroupManager).GetADAMObjectsFromIds( @( $userSettings.DelegateToUser ))
                                $delegation = [PSCustomObject]@{                                                    
                                    UserId = $userObject.Login
                                    FullName = $userObject.Username
                                    DelegatedUserId = $delegatedUserObjects[0].Login
                                    DelegatedFullName = $delegatedUserObjects[0].Fullname
                                }
                            }
                            else
                            {
                                $delegation = [PSCustomObject]@{                                                    
                                    UserId = $userObject.Login
                                    FullName = $userObject.Username
                                    DelegatedUserId = ''
                                    DelegatedFullName = ''
                                }
                            }
                            $delegation
                        }                                            
                    }
                }
            } catch {
                Write-Error $_
                continue
            }
        }
    }

    End
    {
        if ( !$PreserveSession )
        {
            Stop-FileHoldSession -InformationAction $InformationPreference
        }
    }
}

try {
    Export-ModuleMember -Function Get-FileHoldAutoDelegateUser                                 
} catch {
    Write-Warning -Message "Get-FileHoldAutoDelegateUser executing directly."
}

# Load FH API functions. Assume psm1 is in same location as this script.
#    Import-Module "$PSScriptRoot\FileHoldApi.psm1" -Force
#    . "$PSScriptRoot\Get-FileHoldUserObjects.ps1"
#    ''|Get-FileHoldAutoDelegateUser -Server 'http://rtest13' -UserId 'sysadm' -Password '12345' -SearchType ExactName -InformationAction Continue -Verbose
