﻿<#
    #############################################################################################
    .SYNOPSIS
    Copyright(C)2019 FileHold Systems Inc.
    #############################################################################################
#>
$BuildNumber = '20190628.1'

# Load FH API functions. Assume ps1 is in same location as this script.
$ScriptDir = Split-Path -Path $script:MyInvocation.MyCommand.Path
. ( $ScriptDir + '\FileHoldApi-2.ps1' )

function Get-DocumentVersionsSelection
{
  <#
      .SYNOPSIS
        Get a snapshot selection of the given document versions.
      .DESCRIPTION
      .OUTPUTS
        New selection containing document versions.
  #>

    Param (
        [System.Collections.ArrayList]$DocumentVersions
    )

    $snapshotId = [GUID]::Empty
    $searchCriteria = New-Object (LibraryManager.SearchCriteria)
    $searchCriteria.OnlyMetadata = $true

    [System.Collections.ArrayList]$conditions = @()

  # search on list of document version ids
    $i = $conditions.Add( (New-Object (LibraryManager.SearchCondition)) )
    $conditions[$i].SearchType = [DocumentFinder.SearchType]::DocumentVersionId.value__
    $conditions[$i].OperatorType = [DocumentFinder.Operator]::InList.value__
    $conditions[$i].Operands = $DocumentVersions

  # include any document version, but only latest metadata version
    $i = $conditions.Add( (New-Object (LibraryManager.SearchCondition)) )
    $conditions[$i].SearchType = [DocumentFinder.SearchType]::OnlyLastVersion.value__
    $conditions[$i].OperatorType = [DocumentFinder.Operator]::Equal.value__
    $conditions[$i].Operands = @( $false, $true )

  # we do not care how many metadata versions
    $i = $conditions.Add( (New-Object (LibraryManager.SearchCondition)) )
    $conditions[$i].SearchType = [DocumentFinder.SearchType]::ReturnLastForBinaryVersion.value__
    $conditions[$i].OperatorType = [DocumentFinder.Operator]::Equal.value__
    $conditions[$i].Operands = $true

  # there will be some in the archive, but we do not care about those
    $i = $conditions.Add( (New-Object (LibraryManager.SearchCondition)) )
    $conditions[$i].SearchType = [DocumentFinder.SearchType]::IncludeArchive.value__
    $conditions[$i].OperatorType = [DocumentFinder.Operator]::Equal.value__
    $conditions[$i].Operands = $false

    $searchCriteria.SearchConditions = $conditions

    $searchResults = (DocumentFinder -ErrorAction Stop).GetDocumentsBySnapshot( [GUID]::Empty, [ref]$snapshotId, 'SRT', $searchCriteria, $null, 0, 100 )

    $snapshotSelection = New-Object (LibraryManager.SnapshotSelection)
    $snapshotSelection.SnapshotId = $snapshotId
    $snapshotSelection.DocumentIdList = @( $searchResults.DocumentValues.ForEach( { $_.DocumentId } ))
    $snapshotSelection.MetadataVersionidList = @( $searchResults.DocumentValues.ForEach( { $_.MetadataVersionId } ))
    $selection = New-Object (LibraryManager.Selection)
    $selection.SnapshotSelection = @( $snapshotSelection )
    
    (DocumentManager -ErrorAction Stop).CreateSelection( $selection )
}


function Remove-DocumentVersions
{
  <#
      #############################################################################################
      .SYNOPSIS
      Remove document versions

      .DESCRIPTION
      Given a list of document versions in the Powershell pipeline, remove them.

      .PARAMETER Server
      Address of the FileHold server, like http://myserver/fh/filehold
      .PARAMETER UserId
      FileHold user id with a role suitable for this script. If this 
      value and the password are not provided the user will be 
      logged in using integrated authentication.
      .PARAMETER Password
      Password to use with the UserId
      .PARAMETER Domain
      Active directory domain for the user. If this value is not 
      provided the user will be logged in as a local user.

      .NOTES
      Use -WhatIf to perform all functions except updating the field.

      DISCLAIMER. 
    
      FileHold makes no claims to the correctness, fitness for purpose, or
      anything else related to this script. It is provided as an example only.
      It is intended to be used or modified by a person skilled with Windows,
      PowerShell, .NET programming, and the FileHold API. Never use it on a
      production system without thouroughly testing it first and never use it
      in production if you do not fit the skilled person description above.
      #############################################################################################
  #>
  [CmdletBinding(SupportsShouldProcess = $true)]
  Param (
    [Parameter(
        ValueFromPipeline = $true,
        HelpMessage='A list of document versions to remove.',
        Mandatory = $true
    )]
    [Object]$ValueList,
    
    [Parameter(
        Mandatory = $false, 
        HelpMessage='Name or URI for FileHold server. HTTP assumed if protocol not specified.'
    )]
    [uri]$Server = 'localhost',
    
    [string]$UserId = $null,
    
    [string]$Password = $null,
    
    [string]$Domain = $null           
  )
  
  Begin
  {
    Write-Information -MessageData "Remove document version $(Get-Date) ($($MyInvocation.MyCommand.Name) $BuildNumber)"

    if ( $WhatIfPreference ) 
    {
      Write-Verbose -Message 'Simulate mode is enabled. The system will not be updated. Invoke without ""-WhatIf"" to make changes.'
    }
    
    if ( $Domain )
    {
      Start-FileHoldSession -HostAddress $Server -UserId $UserId -Password $Password -Domain $Domain -BuildNumber $BuildNumber
    }
    if ( $UserId -and $Password )
    {
      Start-FileHoldSession -HostAddress $Server -UserId $UserId -Password $Password -BuildNumber $BuildNumber
    }
    else
    {
      Start-FileHoldSession -HostAddress $Server -UseIntegratedAuthentication -BuildNumber $BuildNumber 
    }
      
    $documentVersionIdsToDelete = New-Object -TypeName System.Collections.ArrayList -ArgumentList ( $null )
      
    $totalItems = 0
  }

  Process
  {
    foreach ( $documentVersionId in $ValueList )
    {
      $totalItems += 1
      $null = $documentVersionIdsToDelete.Add( [Int32]$documentVersionId )      
      Write-Information -MessageData "Plan to remove document version id ""$($documentVersionId)""."
    }
  }

  End
  {
    Write-Information -MessageData "Attempt to remove $totalItems document versions."
    $documentVersionsSelection = Get-DocumentVersionsSelection $documentVersionIdsToDelete
    if ( $pscmdlet.ShouldProcess( $documentVersionsSelection, 'Remove document versions') )
    {
      $null = (DocumentManager).DeleteDocuments( $documentVersionsSelection, $false )
    }
    Stop-FileHoldSession
  }
}

$testQuery = "select * from DocumentVersions dvMain
                where dvMain.DocumentId in (
                  SELECT 
                      dv.[DocumentId]
                    FROM [DocumentVersions] dv
                      inner join MetadataVersions mv on mv.DocumentVersionId = dv.DocumentVersionId
                      inner join DocumentSchemas ds on ds.DocumentSchemaId = mv.DocumentSchemaId
                      where ApprovalStatus = 2 
                        and dv.type <> 2 -- ignore offline
                        and dv.islast = 1 -- only compare with the latest version
                        and dv.isdeleted = 0 -- ignore versions that have been deleted
                        and ds.Name = 'Schema-1'
                  )
                  and dvMain.IsLast = 0 
                  and dvMain.IsDeleted = 0"
  
( Invoke-Sqlcmd -Database ch_librarymanager -Query $testQuery ).DocumentVersionId `
    | Remove-DocumentVersions -UserId sysadm -Password 12345 -InformationAction Continue -Verbose
#    | Remove-DocumentVersions -UserId sysadm -Password 12345 -InformationAction Continue -Verbose -WhatIf
