﻿<#
    .SYNOPSIS
        Implement functions for using the FileHold API
        Copyright(C)2017-2019 FileHold Systems Inc.
    .DESCRIPTION
        DISCLAIMER. 
    
        FileHold makes no claims to the correctness, fitness for purpose, or
        anything else related to this script. It is provided as an example only.
        It is intended to be used or modified by a person skilled with Windows,
        PowerShell, .NET programming, and the FileHold API. Never use it on a
        production system without thouroughly testing it first and never use it
        in production if you do not fit the skilled person description above.

        Call FH-Start-Session to begin using the API. After the call to start
        the session, the user will be set to the last referenced user for use
        in later web service calls.

        Make a call to any service using parenthesis to evaluate the function name. 
        The last referenced user will be used for the service call. The last user 
        can be overridden by including the user id as a parameter to the service 
        reference evaluation. This method makes it possible to have more than
        one user logged in at the same time and interacting in a natural way 
        in the script. 
        
        For example, you need to complete some workflow tasks
        in bulk. You create a special user for the purpose of completing the tasks
        but first you need to delegate the tasks to them and for that you 
        need an administrator. You could log in as the administrator and delegate
        all the tasks, then logout, login as the bulk user and complete all 
        the tasks. It might be both easier and more logical to delgate a single
        task as the administrator and then immediately complete it as the bulk
        user. This is no probem using the mutli-user session cache in this script.
        
        Note that Intellisense will not work with this method of accessing services, 
        but as with all Powershell you can perform some interactive tricks to 
        get around this. For example, from the interactive window:

        $dm = DocumentManager

        Then simply type "$dm." and Intellisense will kick in. You can copy and
        past the results to your code.
#>

#region Globals
$global:scriptName = $MyInvocation.MyCommand.Name
$global:fileHoldUrl = $null
$global:lastHost = $null
$global:clientVersion = '15.2'
$global:webServiceTimeout = 500000
$global:sessionCache = @{}
[TimeSpan]$global:cacheTimeout = '00:05:00'
$global:urmSessionManagerProxy = $null
$global:urmWindowsLoginProxy = $null

Set-Variable FILEHOLD_COOKIE_NAME -option Constant "FHLSID" -ErrorAction SilentlyContinue
#endregion Globals

#region UserRoleManager
<#
    .SYNOPSIS
        For internal use. 
    .DESCRIPTION        
        Determines if a new web reference is required. This is used with sessionless
        web services so there is no checking or updating of session cookies.
#>
#region WebReferences
function GetSessionlessServiceProxy( [uri]$service, $proxy, $timeout )
{     
    if ( !$proxy )
    {        
        $proxy = New-WebServiceProxy -Uri $service.OriginalString -UseDefaultCredential -Namespace UserRoleManager
        $proxy.Timeout = $timeout
    }
    $proxy
}

function SessionManager
{
    GetSessionlessServiceProxy ( $global:fileHoldUrl + "/UserRoleManager/SessionManager.asmx?WSDL") $global:urmSessionManagerProxy $global:webServiceTimeout 
}

function WindowsLogin
{
    GetSessionlessServiceProxy ( $global:fileHoldUrl + "/UserRoleManager/WindowsLogin.asmx?WSDL" ) $global:urmWindowsLoginProxy $global:webServiceTimeout
}

function UserRoleManager.Client()
{
    ( (SessionManager).GetType().Namespace + '.Client' )
}
#endregion WebReferences

#region MultiUserSessionCache
function AddSession
{
  <#
      .SYNOPSIS
        Add a new session or replace an existing session in the session cache.
      .DESCRIPTION
        It is assume the session is for a full user. This allows the script to have more than one user logged
        in at the same time.
  #>
    Param( 
        [Parameter( Mandatory = $true )][string]$userId, 
        [Parameter( Mandatory = $true )][string]$sessionId,
        [Parameter( Mandatory = $true )][string]$hostName
    )

    if ( -not $global:lastHost -or $host -eq $global:lastHost )
    {
        $global:sessionCache.Add( $userId, @{ SessionId = $sessionId; CachedTime = (Get-Date); Host = $hostName } )
        $global:lastUserId = $userId
        $global:lastHost =  $hostName
    }
    else
    {
        throw ( 'The session cache can only contain a single FileHold host. It already contains ' + $global:lastHost + ' and there is an attempt to add ' + $hostName + '.' )
    }
}

function FH-Is-Session-Valid( [GUID]$SessionId )
{
    (SessionManager).IsSessionValid( $SessionId, $false )
}

function GetSessionId
{
  <#
      .SYNOPSIS
        Get a valid session id for the given user.
      .DESCRIPTION
        Gets a session from the cache that should be valid, but there is no guarentee. If the cache times out
        it will check the session against the server. If it is valid it will restart the timer for the cache,
        but the server session time will not be affected. 
  #>    
    Param( 
        [Parameter( Mandatory = $true )][string]$userId
    )

    $cachedTime = $global:sessionCache[$userId].CachedTime
    if ( $cachedTime -ne $null )
    {
        if ( (Get-Date) - $cachedTime -gt $global:cacheTimeout )
        {
            if  ( !(FH-Is-Session-Valid $global:sessionCache[$userId].SessionId ) )
            {
                $global:sessionCache.Remove( $userId )
                $null
            }
            else
            {
                $global:sessionCache[$userId].CachedTime = Get-Date
                $global:sessionCache[$userId].SessionId
            }
        }
        else
        {
            $global:sessionCache[$userId].SessionId
        }
        $global:lastUserId = $userId
    }
    else
    {
        $null
    }
}

function RemoveSession()
{
    Param (
        [Parameter(Mandatory = $true)]
        [string]$userId
    )
    $global:sessionCache.Remove( $userId )        
    if ( $global:sessionCache.Count -eq 0 ) { $global:lastHost = $null }
}

function RemoveAllSessions()
{
    $global:sessionCache = @{}        
    $global:lastHost = $null
}

function GetAllSessionIds()
{
    ($global:sessionCache.Keys).ForEach( { $global:sessionCache[$_].SessionId } )
}

function SetCacheTimeout()
{ 
    Param (
        [Parameter(Mandatory = $true)]
        [TimeSpan]$Timeout
    )
    $global:cacheTimeout = $Timeout
}
#endregion MultiUserSessionCache

<#
    Must be called while connection remains "established"
#>
function GetActiveClientAddress
{
    $hostName = ([uri]$global:fileHoldUrl).DnsSafeHost
    $remoteAddresses = Resolve-DnsName -Name $hostName | Select IpAddress

    foreach ( $remoteAddress in $remoteAddresses )
    {
        $localAddress = (((Get-NetTCPConnection)).Where( { $_.OwningProcess -eq $pid } )|? State -eq Established).LocalAddress
    }
    $localAddress
}

function Start-FileHoldSession()
{
  <#
      .SYNOPSIS
        Start a new custom client FileHold session for version 14.2 or higher.

      .DESCRIPTION
        Nothing happens in FileHold without a client session. This function will 
        automatically create the session using integrated Windows authentication,
        direct local user login, or direct domain user login. It also ensures the 
        client IP and script name is provided for the user activity report.

      .EXAMPLE
        PS C:\> FH-Start-Session "http://myserver/FH/FileHold"
      .EXAMPLE 
        PS C:\> FH-Start-Session "http://myserver/FH/FileHold" "myuid" "mypassword"
      .EXAMPLE 
        PS C:\> FH-Start-Session -HostAddress localhost -ShowLogin
      .LINK
        https://www.filehold.com
  #>
    Param (
        [Parameter(Position=1)]
        [string]$HostAddress,
        [Parameter(Mandatory = $false)]
        [string]$UserId,          
        [Parameter(Mandatory = $false)]
        [string]$Password, 
        [Parameter(Mandatory = $false)]
        [string]$Domain = $NULL,
        [Parameter(Mandatory = $false)]
        [switch]$UseIntegratedAuthentication,
        [Parameter(Mandatory = $false)]
        [string]$BuildNumber = $scriptName
    )

  # Future versions of Powershell may have TLS1.2 enabled by default
    [System.Net.ServicePointManager]::SecurityProtocol = [System.Net.SecurityProtocolType]::Tls12;

  # Uncomment the following line to cheat the certificate name check
  #    [System.Net.ServicePointManager]::ServerCertificateValidationCallback = {$true}

  # Allow the caller to provide a partial address and automatically fix it up. Defaults to http and /fh/filehold
    $hostUrl = [uri]$HostAddress
    if ( !$hostUrl.Port ) { $hostUrl = [uri]('http://' + $hostUrl.OriginalString) }
    if ( $hostUrl.AbsolutePath -eq '/' ) { $hostUrl = [uri]($hostUrl.AbsoluteUri + 'fh/filehold') }
    $global:fileHoldUrl = $hostUrl.AbsoluteUri
    
    $serverVersion = $null

    if (( -not $UserId -or -not $Password ) -and -not $UseIntegratedAuthentication )
    {
        $credential = Get-Credential -Message ( 'Password for ' + $HostAddress ) -UserName ( $UserId + (&{if($Domain){'@'+$Domain}}) )
        if ( $credential )
        {
            $Password = $credential.GetNetworkCredential().Password
            $UserId = $credential.GetNetworkCredential().UserName
            $Domain = $credential.GetNetworkCredential().Domain                
        }
        else
        {
            throw 'No authentication credentials were provided.'
        }
    }
    elseif ( $UseIntegratedAuthentication )
    {
    # Extract the userid from the domain\userid        
        $UserId = [System.Security.Principal.WindowsIdentity]::GetCurrent().Name.Split( '\' )[1]
    }

    $sessionId = GetSessionId $UserId

    if ( $sessionId -eq $null )
    {
      if ( $UseIntegratedAuthentication )
      {
        try {
          $sessionId = (WindowsLogin).StartSession( [UserRoleManager.Client]::CustomClient.value__ )
        } catch {
          throw 'Unable to log in using integrated authentication.'
        }
      }
      else
      {
        if ( $Domain )
        {
          $domainId = (((SessionManager).GetStoredDomains()).Where( { $_.Name -ieq $domain } )).Id
          if ( $domainId )
          {
            $sessionId = (SessionManager).StartSessionForDomainUser( $userId, $password, $domainId, [UserRoleManager.Client]::CustomClient.value__ )
          }
          else
          {
            throw 'Domain with the name "' + $Domain + '" was not found.'
          }
        }
        else
        {
          $sessionId = (SessionManager).StartSession( $userId, $password, [UserRoleManager.Client]::CustomClient.value__ )
        }
      }

      if ( $sessionId )
      {            
        $compatible = (SessionManager).CheckApiVersionAndLogClientInfo( $sessionId, `
          $global:clientVersion, [ref]$serverVersion, `
          $BuildNumber, `
          (GetActiveClientAddress) )
    
        $global:hostUrl = $uri.Host

        AddSession $userId $sessionId $global:fileHoldUrl

        $sessionInfo = FH-Session-Info
        Write-Information "Session ""$sessionId"" for user $($sessionInfo.UserName) ($($sessionInfo.UserGuid)) created."
        Write-Verbose "Server version $serverVersion may not be compatible with client version $clientVersion."
      }
    }
    else
    {
      $sessionInfo = FH-Session-Info
      Write-Information "Session ""$sessionId"" for user $($sessionInfo.UserName) ($($sessionInfo.UserGuid)) created."
    }
}

function Stop-FileHoldSession()
{
  <#
      .SYNOPSIS
        Ends the current session. Does nothing if there is no current session.
  #>
    Param (
        [Parameter(Mandatory = $false)]
        [string]$UserId = $NULL
    )
    if ( $userId )
    {        
        try
        {
            (SessionManager).EndSession( (GetSessionId $userId) )
            RemoveSession( $userId )
            Write-Information "User "" $userId "" logged out."
        } catch { throw $PSItem }
    }
    else
    {
        (GetAllSessionIds).ForEach( 
        { 
            (SessionManager).EndSession( $_ ); 
            Write-Information "Session "" $_ "" logged out."
        } )
        RemoveAllSessions
    }
}

function FH-Session-Info( $UserId )
{
  <#
      .SYNOPSIS
        Returns details about the current session.
  #>
    if ( $UserId ) 
    { 
        $sessionId = GetSessionId $UserId
    } 
    else 
    { 
        $sessionId = GetSessionId $global:lastUserId
    }
    (SessionManager).GetSessionInfo( $sessionId )   
}
#endregion UserRoleManager

#region SessionBasedWebReferences
<#
    .SYNOPSIS
        For internal use. 
    .DESCRIPTION        
        Determines if a new web reference is required, 
        if a new session cookie should be added to the current web reference or
        if everthing is already good to go.
#>
function GetFileHoldServiceProxy( [uri]$service, $proxy, [string]$userId, [int]$timeout )
{     
    if ( !$proxy )
    {        
        $proxy = New-WebServiceProxy -Uri $service.OriginalString -Namespace ((Get-PSCallStack)[1].FunctionName)
        $proxy.Timeout = $timeout
        $proxy.CookieContainer = New-Object System.Net.CookieContainer 
        $cookie = New-Object System.Net.Cookie( $FILEHOLD_COOKIE_NAME, ( GetSessionId $userId ), "/", $service.Host ) 
        $proxy.CookieContainer.Add( $cookie )    
    }
    else
    {
        $oldSessionId = ($proxy.CookieContainer.GetCookies( $service ))[$FILEHOLD_COOKIE_NAME]
        if ( $oldSessionId -ne (GetSessionId $userId ))
        {
            $cookie = New-Object System.Net.Cookie( $FILEHOLD_COOKIE_NAME, ( GetSessionId $userId ), "/", $service.Host ) 
            $proxy.CookieContainer.Add( $cookie )    
        }
    }
    $proxy
}

function GetWebRequestSession( [uri]$service, $session, [string]$userId )
{   
    if ( !$session )
    { 
        $session = New-Object Microsoft.PowerShell.Commands.WebRequestSession    
        $cookie = New-Object System.Net.Cookie( $FILEHOLD_COOKIE_NAME, ( GetSessionId $userId ), "/", $service.Host ) 
        $session.Cookies.Add($cookie);
    }
    else
    {
        $oldSessionId = ($session.Cookies.GetCookies( $service ))[$FILEHOLD_COOKIE_NAME]
        if ( $oldSessionId -ne (GetSessionId $userId ))
        {
            $cookie = New-Object System.Net.Cookie( $FILEHOLD_COOKIE_NAME, ( GetSessionId $userId ), "/", $service.Host ) 
            $session.Cookies.Add( $cookie )    
        }
    }
    $session
}
#endregion SessionBasedWebReferences
<#
    .SYNOPSIS
        Upload the give file as a single chunk.
#>
#region DocumentRepository
function UploadDocument()
{
    Param (
        [Parameter(Mandatory = $true)]
        $documentInfo,        
        [Parameter(Mandatory = $true)]
        $fileContents,
        [Parameter(Mandatory = $false)]
        [string]$userId = $NULL
    )
    $uploadHandler = "/DocumentRepository/UploadHandler.ashx"

    $global:webRequestSession = GetWebRequestSession ( $global:fileHoldUrl + $uploadHandler ) `
                                                       $global:webRequestSession `
                                                       ( &{ if ( $userId ) { $userId } else { $global:lastUserId }} )

    $token = (RepositoryController).CreateUploadToken( $fileContents.Length )

    $result = Invoke-WebRequest `
                      -body $fileContents `
                      -Uri ( $global:fileHoldUrl + $uploadHandler + '?token=' + $token ) `
                      -WebSession $global:webRequestSession `
                      -Method Post `
                      -ContentType 'application/octet-stream' 
    if ( $result.StatusCode -ne 200 )
    {
        (RepositoryController).RollbackFileCreation( $token )
        throw 'Error uploading file: ' + $result.StatusCode + ' ' + $result.StatusDescription
    }

    $documentInfo.UploadToken = $token
    (DocumentManager).AddDocumentInfo( $documentInfo )
}

function RepositoryController()
{
    Param (
        [Parameter(Mandatory = $false)]
        [string]$userId = $NULL
    )
    GetFileHoldServiceProxy ( $fileHoldUrl + "/DocumentRepository/RepositoryController.asmx?WSDL" ) `
                            $global:drRepositoryControllerProxy `
                            ( &{ if ( $userId ) { $userId } else { $global:lastUserId }} ) `
                            $global:webServiceTimeout
}
#endregion DocumentRepository
#region LibraryManager
#region ClassHelpers
function LibraryManager.DocumentInfo()
{
    ( (DocumentManager).GetType().Namespace + '.DocumentInfo' )
}

function LibraryManager.FieldWithValue()
{
    ( (DocumentManager).GetType().Namespace + '.FieldWithValue' )
}

function LibraryManager.SearchCriteria()
{
    ( (DocumentFinder).GetType().Namespace + '.SearchCriteria' )
}

function LibraryManager.SearchCondition()
{
    ( (DocumentFinder).GetType().Namespace + '.SearchCondition' )
}

function LibraryManager.SnapshotSelection()
{
    ( (DocumentManager).GetType().Namespace + '.SnapshotSelection' )
}

function LibraryManager.Selection()
{
    ( (DocumentManager).GetType().Namespace + '.Selection' )
}

function LibraryManager.FieldType()
{
  ( (DocumentManager).GetType().Namespace + '.FieldType' )
}

function LibraryManager.DropdownFieldChoice()
{
  ( (DocumentSchemaManager).GetType().Namespace + '.DropdownFieldChoice' )
}
#endregion ClassHelpers
#region WebServices
function LibraryStructureManager()
{
    Param (
        [Parameter(Mandatory = $false)]
        [string]$userId = $NULL
    )
    $global:lmLibraryStructureManagerProxy = `
                            GetFileHoldServiceProxy ( $fileHoldUrl + "/LibraryManager/LibraryStructureManager.asmx?WSDL" ) `
                            $global:lmLibraryStrutureManagerProxy `
                            ( &{ if ( $userId ) { $userId } else { $global:lastUserId }} )`
                            $global:webServiceTimeout
    $global:lmLibraryStructureManagerProxy
}

function DocumentManager()
{
    Param (
        [Parameter(Mandatory = $false)]
        [string]$userId = $NULL
    )
    $global:lmDocumentManagerProxy = `
                            GetFileHoldServiceProxy ( $fileHoldUrl + "/LibraryManager/DocumentManager.asmx?WSDL" ) `
                            $global:lmDocumentManagerProxy `
                            ( &{ if ( $userId ) { $userId } else { $global:lastUserId }} )`
                            $global:webServiceTimeout
    $global:lmDocumentManagerProxy
}

function DocumentFinder()
{
    Param (
        [Parameter(Mandatory = $false)]
        [string]$userId = $NULL
    )
    $global:lmDocumentFinderProxy = `
                            GetFileHoldServiceProxy ( $fileHoldUrl + "/LibraryManager/DocumentFinder.asmx?WSDL" ) `
                            $global:lmDocumentFinderProxy `
                            ( &{ if ( $userId ) { $userId } else { $global:lastUserId }} )`
                            $global:webServiceTimeout

    $global:lmDocumentFinderProxy
}


function VirtualFolderManager()
{
    Param (
        [Parameter(Mandatory = $false)]
        [string]$userId = $NULL
    )
    GetFileHoldServiceProxy ( $fileHoldUrl + "/LibraryManager/VirtualFolderManager.asmx?WSDL" ) `
                            $global:lmVirtualFolderManager `
                            ( &{ if ( $userId ) { $userId } else { $global:lastUserId }} )`
                            $global:webServiceTimeout
}

function DocumentSchemaManager()
{
    Param (
        [Parameter(Mandatory = $false)]
        [string]$userId = $NULL
    )
    $global:lmDocumentSchemaManager = GetFileHoldServiceProxy `
                            ( $fileHoldUrl + "/LibraryManager/DocumentSchemaManager.asmx?WSDL" ) `
                            $global:lmDocumentSchemaManager `
                            ( &{ if ( $userId ) { $userId } else { $global:lastUserId }} )`
                            $global:webServiceTimeout
    $global:lmDocumentSchemaManager
}

function UserPreferences()
{
    Param (
        [Parameter(Mandatory = $false)]
        [string]$userId = $NULL
    )
    $global:lmUserPreferences = GetFileHoldServiceProxy `
                            ( $fileHoldUrl + "/LibraryManager/UserPreferences.asmx?WSDL" ) `
                            $global:lmUserPreferences `
                            ( &{ if ( $userId ) { $userId } else { $global:lastUserId }} )`
                            $global:webServiceTimeout
    $global:lmUserPreferences
}

function DocumentFinder()
{
    Param (
        [Parameter(Mandatory = $false)]
        [string]$userId = $NULL
    )
    $global:lmDocumentFinder = GetFileHoldServiceProxy `
                            ( $fileHoldUrl + "/LibraryManager/DocumentFinder.asmx?WSDL" ) `
                            $global:lmDocumentFinder `
                            ( &{ if ( $userId ) { $userId } else { $global:lastUserId }} )`
                            $global:webServiceTimeout
    $global:lmDocumentFinder
}
#endregion WebServices
#endregion LibraryManager