﻿<#
    Approve tasks in FileHold from a list of task GUIDs
    Copyright(C)2017 FileHold Systems Inc.

    DISCLAIMER. 
    
    FileHold makes no claims to the correctness, fitness for purpose, or
    anything else related to this script. It is provided as an example only.
    It is intended to be used or modified by a person skilled with Windows,
    PowerShell, .NET programming, and the FileHold API. Never use it on a
    production system without thouroughly testing it first and never use it
    in production if you do not fit the skilled person description above.
#>

# We add some DLL references in order to get access to certain constant values needed for the FHAPI
Add-Type -Path 'C:\Program Files\FileHold Systems\Application Server\fileholdadm\FileHold.Common.dll'

$global:scriptName = $MyInvocation.MyCommand.Name
$global:clientAddress = (get-netadapter | get-netipaddress | ? addressfamily -eq 'IPv4').ipaddress
$global:fileHoldUrl = $null
$global:hostUrl = $null
$global:cookie = $null
$global:sessionId = $null
$global:urmSessionManagerProxy = $null
$global:lmWorkflowManagerProxy = $null

Set-Variable FILEHOLD_COOKIE_NAME -option Constant "FHLSID" -ErrorAction SilentlyContinue
Set-Variable NULL_GUID -option Constant "00000000-0000-0000-0000-000000000000" -ErrorAction SilentlyContinue

function FH-Start-Session()
{
 <#
    .SYNOPSIS
        Start a new custom client FileHold session for version 14.2 or higher.
    .DESCRIPTION
        Nothing happens in FileHold without a client session. This function will 
        automatically create the session using integrated Windows authentication,
        direct local user login, or direct domain user login. It also ensures the 
        client IP and script name is provided for the user activity report.

        Sets some global variables:
            fileHoldUrl
            sessionId
            hostUrl
    .EXAMPLE
        PS C:\> FH-Start-Session "http://myserver/FH/FileHold"
    .EXAMPLE 
        PS C:\> FH-Start-Session "http://myserver/FH/FileHold" "myuid" "mypassword"
    .LINK
        https://www.filehold.com
  #>
    Param (
        [Parameter(Position=1)]
        [string]$HostAddress,
        [Parameter(Mandatory = $false)]
        [string]$UserId = $NULL,          
        [Parameter(Mandatory = $false)]
        [string]$Password = $NULL, 
        [Parameter(Mandatory = $false)]
        [string]$Domain = $NULL
    )

    $clientVersion = "14.2"
    $global:fileHoldUrl = $hostAddress
    $serverVersion = ""
    $UseIWA = ( $UserId -eq $NULL ) -and ( $Password -eq $NULL )
    
    Write-Host 'Host' $HostAddress 'UseIWA' $useIWA $userId $password $domain
    if ( $useIWA )
    {
        $uri = $global:fileHoldUrl + "/UserRoleManager/WindowsLogin.asmx?WSDL"
        $urmWindowsLoginProxy = New-WebServiceProxy -Uri $uri -ErrorAction Stop
        $sessionId = $urmWindowsLoginProxy.StartSession( $clientType )
    }
    elseif ( $domain )
    {
        $uri = $glboal:fileHoldUrl + "/UserRoleManager/SessionManager.asmx?WSDL"
        $urmSessionManagerProxy = New-WebServiceProxy -Uri $uri

        $storedDomains = $urmSessionManagerProxy.GetStoredDomains()
        $domainId = $null
        foreach ( $storedDomain in $storedDomains )
        {
            if ( $storedDomain.Name -imatch $domain )
            {
                $domainId = $storedDomain.Id
            }
        }

        if ( $domainId )
        {
            $global:sessionId = $urmSessionManagerProxy.StartSessionForDomainUser( $userId, $password, $domainId, $clientType )
        }
        else
        {
            throw 'Domain with the name "' + $domain + '" was not found.'
        }
    }
    else
    {
        $uri = $global:fileHoldUrl + "/UserRoleManager/SessionManager.asmx?WSDL"
        $urmSessionManagerProxy = New-WebServiceProxy -Uri $uri
        $global:sessionId = $urmSessionManagerProxy.StartSession( $userId, $password, [FileHold.Common.Client]::CustomClient.value__ )
    }

    $compatible = $urmSessionManagerProxy.CheckApiVersionAndLogClientInfo( $sessionId, `
                    $clientVersion, [ref]$serverVersion, `
                    $scriptName, `
                    $clientAddress )
    if ( $compatible -ne 0 )
    {
        'Server version "'+ $serverVersion + '" may not be compatible.' | Write-Warning 
    }
    else
    {
        'Logged in with session GUID "' + $global:sessionId + '"'
    }
    
    $uri = [System.Uri]$urmSessionManagerProxy.Url
    $global:hostUrl = $uri.Host
}

function FH-End-Session
{
<#
    .SYNOPSIS
        Ends the current session. Does nothing if there is no current session.
#>
    if ( $global:sessionId -ne $NULL )
    {
        if ( !$urmSessionManagerProxy )
        {
            $uri = $global:fileHoldUrl + "/UserRoleManager/SessionManager.asmx?WSDL"
            $urmSessionManagerProxy = New-WebServiceProxy -Uri $uri
        }
        try {
            $urmSessionManagerProxy.EndSession( $global:sessionId )
            'Session "' + $global:sessionId + '" logged out.'
            $global:sessionId = $null
        } catch { throw $PSItem }
    }
}

function FH-Session-Info
{
<#
    .SYNOPSIS
        Returns details about the current session.
#>
    if ( !$urmSessionManagerProxy )
    {
        $uri = $global:fileHoldUrl + "/UserRoleManager/SessionManager.asmx?WSDL"
        $urmSessionManagerProxy = New-WebServiceProxy -Uri $uri
    }
    $urmSessionManagerProxy.GetSessionInfo( $global:sessionId )   
}

function FH-Approve-Task( $taskGuid, $workflowInstanceGuid, $comments, $password )
{   
<#
    .SYNOPSIS
        Approve the task as the logged in user.
#>
    $taskStatus = ""

    if ( !$lmWorkflowManagerProxy )
    {
        $uri = $fileHoldUrl + "/LibraryManager/WorkflowManager.asmx?WSDL"
        $lmWorkflowManagerProxy = New-WebServiceProxy -Uri $uri
        $lmWorkflowManagerProxy.CookieContainer = New-Object System.Net.CookieContainer
        $cookie = New-Object System.Net.Cookie( $FILEHOLD_COOKIE_NAME, $global:sessionId, "/", $global:hostUrl ) 
        $lmWorkflowManagerProxy.CookieContainer.Add( $cookie )    
    }

    $workflowGuid = $lmWorkflowManagerProxy.GetTaskWorkflowAndState( $taskGuid, [ref]$taskStatus )
    if ( $workflowGuid -eq $workflowInstanceGuid )
    {
        if ( $taskStatus -eq [FileHold.Common.WorkflowEngine.TaskApprovalStatus]::Undetermined.value__ )
        {
# Approve task
            try {
                $lmWorkflowManagerProxy.RaiseTaskCompletedEvent( $workflowGuid `
                                                                ,$taskGuid `
                                                                ,[FileHold.Common.WorkflowEngine.TaskApprovalStatus]::Approved.value__ `
                                                                ,$comments `
                                                                ,$NULL_GUID  `
                                                                ,$NULL `
                                                                ,$password )
                "Task " + $taskGuid + " for workflow " + $workflowGuid + " has been approved."
            } catch {
                throw $PSItem
            }
        }
        else
        {
            "Task " + $taskGuid + " has status " + $taskStatus + " and is not pending."
        }
    }
    else
    {
        'Task workflow "' + $workflowGuid + '" does not match expected workflow instance GUID "' + $workflowInstanceGuid + '"'
    }
}

function FH-Delegate-Task( $taskGuid, $workflowInstanceGuid, $delegate )
{   
<#
    .SYNOPSIS
        Delegate the task to the given user.
#>
    $taskStatus = ""

    if ( !$lmWorkflowManagerProxy )
    {
        $uri = $fileHoldUrl + "/LibraryManager/WorkflowManager.asmx?WSDL"
        $lmWorkflowManagerProxy = New-WebServiceProxy -Uri $uri
        $lmWorkflowManagerProxy.CookieContainer = New-Object System.Net.CookieContainer
        $cookie = New-Object System.Net.Cookie( $FILEHOLD_COOKIE_NAME, $global:sessionId, "/", $global:hostUrl ) 
        $lmWorkflowManagerProxy.CookieContainer.Add( $cookie )    
    }

    $workflowGuid = $lmWorkflowManagerProxy.GetTaskWorkflowAndState( $taskGuid, [ref]$taskStatus )
    if ( $workflowGuid -eq $workflowInstanceGuid )
    {
        if ( $taskStatus -eq [FileHold.Common.WorkflowEngine.TaskApprovalStatus]::Undetermined.value__ )
        {
            try {
                $lmWorkflowManagerProxy.RaiseTaskDelegated( $workflowGuid `
                                                           ,$taskGuid `
                                                           ,$delegate )

                "Task " + $taskGuid + " for workflow " + $workflowGuid + " has been delegated."
            } catch {
                throw $PSItem
            }
        }
        else
        {
            "Task " + $taskGuid + " has status " + $taskStatus + " and is not pending."
        }
    }
    else
    {
        'Task workflow "' + $workflowGuid + '" does not match expected workflow instance GUID "' + $workflowInstanceGuid + '"'
    }
}

<#
 #############################################################################################

 Main code

 Delegate tasks to the bulk approver user and then log in as the bulk approver and approve
 the tasks. Make sure the bulk approver user has the rights to all the doucment in the 
 workflows or the delegation will fail. The list of tasks is in 'TaskList.txt'. Basic
 exception processing in place to handle login failurs.

 #############################################################################################
#>

$userId = "sysadm"
$password = "12345"
$bulkUser = "bulkapprover"
$bulkUserPassword = "123456"

'Getting bulk user guid'
try {
    FH-Start-Session "http://localhost/fh/filehold" -UserId $bulkUser -Password $bulkUserPassword
    $bulkUserGuid = (FH-Session-Info).UserGuid
} catch {}

if ( $sessionId -ne $null )
{
    FH-End-Session
    $tasks = Import-Csv TaskList.txt -Delimiter `t

    'Beginning delegate phase'
    try {
        FH-Start-Session "http://localhost/fh/filehold" -UserId $userId -Password $password
    } catch {}

    if ( $sessionId -ne $null )
    {
        foreach ( $task in $tasks )
        {
            FH-Delegate-Task $task.TaskGuid `
                             $task.WorkflowInstanceGuid `
                             $bulkUserGuid
        }
        FH-End-Session

        'Beginning approve phase'
        try {
            FH-Start-Session "http://localhost/fh/filehold" -UserId $bulkUser -Password $bulkUserPassword
        } catch {}

        if ( $sessionId -ne $null )
        {
            foreach ( $task in $tasks )
            {
                FH-Approve-Task $task.TaskGuid `
                                $task.WorkflowInstanceGuid `
                                ( "Bulk approval. Ticket 88279. See Fred Smith. Task " + $task.TaskGuid + " workflow " + $task.WorkflowInstanceGuid ) `
                                $bulkUserPassword
            }
            FH-End-Session
        }
        else
        {
            'Unable to login user "' + $bulkuser + '"'| Write-Error
        }
    }
    else
    {
        'Unable to login user "' + $userId + '"'| Write-Error
    }
}
else
{
    'Unable to get bulk user guid.' | Write-Error
}
FH-End-Session